--========================================================--
-- Unseasonal Weather - Radio Forecasting (Server relay)
-- Lee ModData pendiente y emite via DynamicRadio (lado servidor).
--========================================================--

if not UW_RF then UW_RF = {} end

require("UW_RF_State")
require("UW_RF_Messages")

local UW_RF_MD_KEY = "UW_RF_PENDING"

local function ensureChannel(scriptManager)
    -- reusar helper si existe
    if UW_RF_EnsureChannel then
        return UW_RF_EnsureChannel(scriptManager)
    end

    if not DynamicRadio then return nil end
    DynamicRadio.channels = DynamicRadio.channels or {}
    DynamicRadio.cache    = DynamicRadio.cache or {}

    local uuid = "UW-RF-001"
    local freq = 92700

    if DynamicRadio.cache[uuid] then
        return DynamicRadio.cache[uuid]
    end

    local found = false
    for _, ch in ipairs(DynamicRadio.channels) do
        if ch.uuid == uuid then found = true break end
    end
    if not found then
        table.insert(DynamicRadio.channels, {
            name     = "Unseasonal Weather Network",
            freq     = freq,
            category = "Emergency",
            uuid     = uuid,
            register = true,
        })
    end

    if scriptManager and DynamicRadioChannel and DynamicRadioChannel.new then
        local cat = ChannelCategory and ChannelCategory.Emergency or ChannelCategory.Other
        local dynamicChannel = DynamicRadioChannel.new("Unseasonal Weather Network", freq, cat, uuid)
        if dynamicChannel then
            if scriptManager.AddChannel then
                scriptManager:AddChannel(dynamicChannel, false)
            end
            DynamicRadio.cache[uuid] = dynamicChannel
            return dynamicChannel
        end
    end

    return DynamicRadio.cache[uuid]
end

local function emitBroadcast(lines)
    if not lines or #lines == 0 then
        print("[UW_RF][SRV] No lines to broadcast.")
        return
    end

    local channel = ensureChannel()
    if not channel then
        print("[UW_RF][SRV] Channel UW-RF-001 not ready.")
        return
    end

    local bc = RadioBroadCast.new("UW Forecast - SRV:" .. tostring(ZombRand(1, 99999)), -1, -1)
    for _, line in ipairs(lines) do
        if line then
            if line.setCodes then line:setCodes(false) end
            if line.setPriority then line:setPriority(5) end
            if line.setLoop then line:setLoop(false) end
            bc:AddRadioLine(line)
        end
    end

    if bc.getRadioLines and bc:getRadioLines() and bc:getRadioLines():isEmpty() then
        print("[UW_RF][SRV] Broadcast has zero lines, skipping emit.")
        return
    end

    channel:setAiringBroadcast(bc)
    print(string.format("[UW_RF][SRV] Broadcast emitted (%d lines) on UW-RF-001", #lines))
end

local function tryBroadcastServer()
    local md = ModData.getOrCreate(UW_RF_MD_KEY)
    local forecast = md.pending
    if not forecast then return end

    local gt = getGameTime()
    local nowHours = gt and gt:getWorldAgeHours() or 0

    if forecast.expires and nowHours > forecast.expires then
        md.pending = nil
        ModData.transmit(UW_RF_MD_KEY)
        return
    end

    if not forecast.jitter then
        forecast.jitter = ZombRand(1, 6)
        md.pending = forecast
        ModData.transmit(UW_RF_MD_KEY)
    end

    local targetHour = forecast.hour or 0
    local scheduledMinute = forecast.minute or 0
    local jitter = forecast.jitter or 0
    local startMinuteAbs = (targetHour * 60) + scheduledMinute + jitter
    if startMinuteAbs < 0 then startMinuteAbs = 0 end
    local windowMinutes = 30
    local endMinuteAbs = startMinuteAbs + windowMinutes

    local currentHour   = gt and gt:getHour() or 0
    local currentMinute = gt and gt:getMinutes() or 0
    local currentAbs = (currentHour * 60) + currentMinute
    local wrap = 24 * 60
    startMinuteAbs = startMinuteAbs % wrap
    endMinuteAbs   = endMinuteAbs % wrap
    currentAbs     = currentAbs % wrap

    local withinWindow = false
    if startMinuteAbs <= endMinuteAbs then
        withinWindow = currentAbs >= startMinuteAbs and currentAbs <= endMinuteAbs
    else
        withinWindow = currentAbs >= startMinuteAbs or currentAbs <= endMinuteAbs
    end
    if not withinWindow then
        return
    end

    local lines = nil
    if forecast.key and UW_RF_GetLinesForKey then
        lines = UW_RF_GetLinesForKey(forecast.key)
    end
    if not lines then
        local text = forecast.text or forecast.key or "UW_RF forecast"
        local line = RadioLine and RadioLine.new and RadioLine.new(text, 150,150,150) or nil
        if line then
            lines = { line }
        end
    end

    if lines and #lines > 0 then
        emitBroadcast(lines)
    else
        print(string.format("[UW_RF][SRV] No lines for key=%s", tostring(forecast.key)))
    end

    md.pending = nil
    ModData.transmit(UW_RF_MD_KEY)
end

Events.OnLoadRadioScripts.Add(ensureChannel)
Events.EveryOneMinute.Add(tryBroadcastServer)
